/*
 * VerifyAddress.java
 *
 * @Author Alton Blake
 * @Version 1.0 May 10, 2011, 11:38 AM
 */

package clnaddr.servlets;

/* Generic java stuff */
import clnaddr.ws.*;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.List;
import java.util.Properties;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Holder;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class VerifyAddress extends HttpServlet {
    
    private static int      maxConnectRetries=5;
    private String          hostName;
    private String          hostPort;
    private URL             url;
    private CLNBatch        service;
    private CLNBatchSoap    port;
    private BindingProvider bp;
    private Properties      properties;
    private static long     departmentID = 1;    
    private long            customerID;
    private String          password;
    private int             ErrorSuggestCount;
    
    /**
     * Servlet initialization, loads runtime configuration parameters and tests connection to web service server
     *
     * @param config system configuration
     */ 
    @Override
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        try {
            InputStream inputStream = ClassLoader.getSystemResourceAsStream("WEB-INF/clnaddr.properties");
            properties = new Properties();
            String pathToPropertiesFile = config.getServletContext().getRealPath("")+"/WEB-INF/clnaddr.properties";
            properties.load(new FileInputStream(pathToPropertiesFile));
            hostName   = properties.getProperty("clnaddrws.serverHost");
            hostPort   = properties.getProperty("clnaddrws.serverPort");
            customerID = Long.parseLong(properties.getProperty("WebServiceServer.CustomerID"));
            password   = properties.getProperty("WebServiceServer.Password");             
        } catch (IOException e) {
            getServletContext().log("FATAL - Failed to load startup configuration from properties file...\n"+e.toString());
        }

        try{
           ErrorSuggestCount = Integer.parseInt(properties.getProperty("parameters.ErrorSuggestCount"));  
        } catch (Exception e){
            ErrorSuggestCount=10;
        }
        
        /*
        * 1. Create service, and port objects
        */
        getServletContext().log("INFO - Creating service...\n");
        try{
           service = new CLNBatch();
        } catch (Exception e){
             getServletContext().log("ERROR - Failed to instantiate service CLNBatch(url,qname)...\n"+e.toString());
        }
        
        try {
            port    = service.getCLNBatchSoap();
        } catch (Exception e) {
             getServletContext().log("ERROR - Failed to load soap object CLNBatchSoap()...\n"+e.toString());
        }
        
        /*
        * 2. Instantiate service, and port objects using runtime
        *    parameters
        */
        getServletContext().log("INFO - Instantiating service...\n");
        if(port==null){
           getServletContext().log("ERROR - Port was not set. Possible failure instantiating CLNBATCHSoap endpoint...\n"); 
        }
        
        try {
            // Specify port at runtime
            bp = (BindingProvider) port;
            // Set runtime connection parameters
            bp.getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY,"http://"+hostName+":"+hostPort+ "/CLNVerify");
        } catch (Exception e) {
            getServletContext().log("ERROR - Failed to set binding properties...\n"+e.toString());
        }
        
        try{
            testConnection();
            getServletContext().log("INFO - Servlet initialized; connection tested and PASSED; ready to go...\n");
        } catch (Exception e){
           getServletContext().log("ERROR - Servlet testConnection FAILED, there might be a problem...\n"+e.toString());
        }
    }
    /**
     * Destroy's the servlet 
     *
     */    
    @Override
    public void destroy() {
        getServletContext().log("INFO - clnaddr servlet being destroyed...");
    }
    
    /**
     * Servlet get method
     *
     * @param request  servlet request object
     * @param response servlet resonse object
     */     
    @Override
    public void doGet(HttpServletRequest request,HttpServletResponse response) throws IOException, ServletException {        

        Address address = new Address();    
        address.setCompany(request.getParameter("company"));
        address.setAddressLine1(request.getParameter("address1"));
        address.setAddressLine2(request.getParameter("address2"));
        address.setAddressLine3(request.getParameter("address3"));
        address.setCity(request.getParameter("city"));
        address.setState(request.getParameter( "state"));
        address.setZip9(request.getParameter("postalcode"));
        address.setCountyName(request.getParameter("county"));
        address.setCountryCode(request.getParameter("countrycode"));     
        // Perform web service call to CLEAN_Address web service server
        try{
            address =  clnVerifyAddress(address);
        } catch (Exception e)    {
           getServletContext().log("In doGet clnVerifyAddress raised exception...");
           address.errorString=address.errorString+" clnVerifyAddress raised exception...";
        }                
        StringBuffer sb = generateResponse(address);
        response.setContentType("text/xml");
        response.setHeader("Cache-Control","no-cache");
        response.getWriter().write(sb.toString());                
    }
    
    /**
     * Servlet post method
     *
     * @param request  servlet request object
     * @param response servlet resonse object
     */      
    @Override
    public void doPost(HttpServletRequest request,HttpServletResponse response) throws IOException, ServletException {
        ;// Not implemented
    }
    
    /**
     * Builds and transmits SOAP request to the clnaddrws web service server using Java API for XML web services (JAXWS)
     *
     * @param _address  address object containing address to be verified
     * @return Address  verified address object to be returned to requestor
     */      
    public Address clnVerifyAddress(Address _address){
        /*
        customerID        = Long.parseLong(properties.getProperty("WebServiceServer.CustomerID"));
        password          = properties.getProperty("WebServiceServer.Password"); 
        ErrorSuggestCount = properties.getProperty("parameters.ErrorSuggestCount");        
        */
        
        /*
         * 3. Set authentication credentials
         *
         */
        Authentication authentication = new Authentication();
        try {
            authentication.setCustomerID(customerID);
        } catch (Exception e) {
            getServletContext().log("While setting Customer ID in authentication object...\n"+e.toString());// Do something
        }
        
        try {
            authentication.setDepartmentID(departmentID);
        } catch (Exception e) {
            getServletContext().log("While setting Department ID in authentication object...\n"+e.toString());// Do something
        }
        authentication.setPassword(password);
        
        /*
         * 4. Set address verification parameters
         *
         */
        Parameters parameters = new Parameters();
        parameters.setVerifyAddress(true);
        parameters.setGeoCodeAddress(true);
        parameters.setCheckTelephone(true);
        parameters.setParseName(true);
        parameters.setParseEmail(true);
        parameters.setDNSCheckEmail(false);
        parameters.setUseGenericAddressLines(true);
        parameters.setGenericAddressLines(3);
        parameters.setFullNameInAddress(false);
        parameters.setCompanyNameInAddress(false);
        parameters.setCountryReturnType(CountryReturnType.fromValue("ISO_2"));
        parameters.setPreferredLanguage(PreferredLanguageType.fromValue("DEFAULT"));
        parameters.setErrorSuggestCount(ErrorSuggestCount); 
        parameters.setVerboseSuggestList(true);
        parameters.setIncludeSuggestHeader(true);
        parameters.setMaxCityLength(30);
        parameters.setMaxProvinceLength(30);
        parameters.setMaxAddressLineLength(40);
        parameters.setExpandStreetSuffix(false);
        parameters.setExpandSingleDirectional(false);
        parameters.setExpandDoubleDirectional(false);
        parameters.setSuiteLineBeforeStreet(false);
        parameters.setAppendSuiteToStreet(true);
        parameters.setOutputAddressWithCompanyNameAttention(false);
        parameters.setAlwaysStandardize(true);
        parameters.setAlwaysUpperCase(false);
        parameters.setBasicAddressSearch(false);
        parameters.setPhoneNumberOutputFormat("");
        
        /*
         * 5. Create address object to be verified
         *
         */
        AddressIn addressIn = new AddressIn();
        addressIn.setCompany(_address.getCompany());
        addressIn.setAddressLine1(_address.getAddressLine1());
        addressIn.setAddressLine2(_address.getAddressLine2());
        addressIn.setAddressLine3(_address.getAddressLine3());
        //addressIn.setSuite(_address.getSuite());
        addressIn.setCity(_address.getCity());
        addressIn.setState(_address.getState());
        addressIn.setPostalCode(_address.getZip9());
        addressIn.setCountryCode(_address.getCountryCode());
        ArrayOfAddressIn arrayOfAddressIn = new ArrayOfAddressIn();
        arrayOfAddressIn.getAddressIn().add(addressIn);
        
        /*
         * 6. Construct request object
         *
         *    Note: you will need to RTI CLNVerify.java class
         */
        CLNVerify req = new CLNVerify();
        req.setAddressCount(arrayOfAddressIn.getAddressIn().size());
        req.setAuthentication(authentication);
        req.setCampaignID("");
        req.setAddressesIn(arrayOfAddressIn);
        req.setParameters(parameters);

        /*
         * 7. Instantiate and populate holders for user defined and
         *    complex data types
         */
        Holder<String> campaign_id = new Holder<String>(req.getCampaignID());
        Holder<Long> error_code = new Holder<Long>();
        Holder<String> error_message = new Holder<String>();
        Holder<Long> result_count = new Holder<Long>();
        Holder<ArrayOfAddressOut> array_of_address_out = new Holder<ArrayOfAddressOut>();
        
        /*
         * 8. Invoke service operation
         *
         */
        try{
            port.clnVerify(req.getAuthentication()
            , new Holder<String>( req.getCampaignID())
            , req.getParameters()
            , req.getAddressCount()
            , req.getAddressesIn()
            , error_code
                    , error_message
                    , result_count
                    , array_of_address_out
                    );
        } catch (Exception e){
            getServletContext().log("In doGet clnVerifyAddress raised exception...\n"+e.toString());
        }
        
        /*
         * 9. Process error codes, messages and results counts
         *
         */
        
        ArrayOfAddressOut retval = array_of_address_out.value;
        List<AddressOut> addressOut_list = retval.getAddressOut();
        
        /*
         * 10. Process verified address response
         *
         */
        for (AddressOut addressOut : addressOut_list) {
            _address.setCompany(nullToString(addressOut.getCompany()));
            _address.setAddressLine1(nullToString(addressOut.getAddressLine1()));
            _address.setAddressLine2(nullToString(addressOut.getAddressLine2()));
            _address.setAddressLine3(nullToString(addressOut.getAddressLine3()));
            _address.setSuite(nullToString(addressOut.getSuite()));
            _address.setCity(nullToString(addressOut.getCity()));
            _address.setState(nullToString(addressOut.getState()));
            _address.setZip9(nullToString(addressOut.getPostalCodeFull()));
            _address.setCountyFips(nullToString(addressOut.getCountyFIPS()));            
            _address.setCountyName(nullToString(addressOut.getCountyName()));
            _address.setCountryCode(nullToString(addressOut.getCountryCode()));
            _address.setCountryName(nullToString(addressOut.getCountryName()));            
            _address.setLatitude(nullToString(addressOut.getLatitude()));
            _address.setLongitude(nullToString(addressOut.getLongitude()));            
            _address.statusCode  = nullToString(addressOut.getStatusCode()).trim();
            _address.errorCode   = nullToString(addressOut.getDPVErrorCode()).trim();
            _address.errorString = nullToString(addressOut.getErrorString());                        
            _address.setSuggestionList( nullToString(addressOut.getSuggestionList()));            
            _address.setDeliveryPointCode( nullToString(addressOut.getDeliveryPoint()));
            _address.setAddressKey(nullToString(addressOut.getAddressKey()));    
            System.out.println("_address.getSuggestionList()="+_address.getSuggestionList());
        }

        return _address;
    }
    
    /**
     * Builds an XML response document from the address verification results
     *
     * @param _address  address object containing  address verification results
     * @return StringBuffer string buffer containing XML response
     */      
    private StringBuffer generateResponse(Address _address){
        /* Generate and return response object */
        StringBuffer sb = new StringBuffer();
        sb.append("<?xml version=\"1.0\"?>");
        sb.append("<address language=\"en_US\">");
        sb.append("<company>"+nullToString(_address.getCompany())+"</company>");        
        sb.append("<address1>"+nullToString(_address.getAddressLine1())+"</address1>");
        sb.append("<address2>"+nullToString(_address.getAddressLine2())+"</address2>");
        sb.append("<address3>"+nullToString(_address.getAddressLine3())+"</address3>");
        sb.append("<city>"+nullToString(_address.getCity())+"</city>");
        sb.append("<state>"+nullToString(_address.getState())+"</state>");
        sb.append("<postalcode>"+nullToString(_address.getZip9())+"</postalcode>");        
        sb.append("<countyfips>"+nullToString(_address.getCountyFips())+"</countyfips>");  
        sb.append("<county>"+nullToString(_address.getCountyName())+"</county>");       
        sb.append("<countrycode>"+nullToString(_address.getCountryCode())+"</countrycode>");
        sb.append("<countryname>"+nullToString(_address.getCountryName())+"</countryname>");        
        sb.append("<latitude>"+nullToString(_address.getLatitude())+"</latitude>");
        sb.append("<longitude>"+_address.getLongitude()+"</longitude>");        
        sb.append("<statuscode>"+nullToString(_address.statusCode)+"</statuscode>");
        sb.append("<dpverrorcode>"+nullToString(_address.errorCode)+"</dpverrorcode>");
        sb.append("<errorstring>"+nullToString(_address.errorString)+"</errorstring>");
        sb.append("<deliverypoint>"+nullToString(_address.getDeliveryPointCode())+"</deliverypoint>");     
        sb.append("<addresskey>"+nullToString(_address.getAddressKey())+"</addresskey>");  
        StringBuffer suggestions = new StringBuffer();
        String[]     suggestionsArray = nullToString(_address.getSuggestionList()).split(";");
        int row=0;
        for (int i = 0; i < suggestionsArray.length; i++){
           if (suggestionsArray[i].trim().length() > 0){
             //suggestions.append("<li id='clnaddrsli-"+(row)+"' ><a id='clnaddrslr-"+(row)+"' ondblclick='processDblClick(this.id)' class='"+(((i%2)==0) ? "even" : "odd" )+"'>"+suggestionsArray[i].replace(":"," ")+"</a></li>");			
             suggestions.append("<li id='clnaddrsli-"+(row)+"' ><a id='clnaddrslr-"+(row)+"' ondblclick='processDblClick(this.id)' class='"+(((i%2)==0) ? "even" : "odd" )+"'>"+suggestionsArray[i]+"</a></li>");			               
             row++;             
           }
          
        }                            
        sb.append("<suggestionlist>");
        if(suggestions.length() > 0){
            sb.append("<![CDATA[<ul id='clnaddrsul' style='list-style: none;margin:0;padding:0;' class='treemenu'>"+suggestions+"</ul>]]>");
        }        
        sb.append("</suggestionlist>");                
        sb.append("<transmitStatus>0</transmitStatus>");
        sb.append("</address>");
        System.out.println(sb.toString());
        suggestions=null;
        return sb;
    }
    /**
     * Extracts each node's value, if error or no value, returns blank (not null)
     *
     * @param elLists  XML document element list
     * @param sTagname XML document tag name
     * @return String  value extracted from XML node
     */     
    private String NodeValue(Element elLists, String sTagName) {
        String sNodeValue = "";
        try {
            NodeList nlXml = elLists.getElementsByTagName(sTagName);
            Element elXml = (Element)nlXml.item(0);
            NodeList nlXmlValue = elXml.getChildNodes();
            if (((Node)nlXmlValue.item(0))!=null){
                sNodeValue = ((Node)nlXmlValue.item(0)).getNodeValue().trim();
            }
        } catch (Exception e){
            sNodeValue = "";
        }
        return sNodeValue;
    }
    
    /**
     * Convert all null characters to string. why? Response doesn't like nulls
     *
     * @param theString  value to be evaluated
     * @return String  theString value to be returned or "" instead of null
     */     
    public String nullToString(String theString) {
        String returnString = "";
        if (theString != null) {
            returnString = theString;
        }
        return returnString;
    }
    
    /**
     * Tests connectivity to clnaddrws SOAP services via Java API for XML Web Services (JAXWS)
     *
     */  
    public void testConnection(){
        
        CLNVerify req = new CLNVerify();
        port.clnVerify(new Authentication(),
                new Holder<String>(""),
                req.getParameters(),
                req.getAddressCount(),
                req.getAddressesIn(),
                new Holder<Long>(),
                new Holder<String>(),
                new Holder<Long>(),
                new Holder<ArrayOfAddressOut>());
    }
    
}
